<?php
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');

require_once '../db/functions.php'; // فایل اتصال به دیتابیس
require_once '../config.php';        // شامل JWT_SECRET و تنظیمات

// تابع ساخت توکن JWT با PHP خام
function base64UrlEncode($data) {
    return rtrim(strtr(base64_encode($data), '+/', '-_'), '=');
}

function createJWT($payload, $secret) {
    $header = json_encode(['alg' => 'HS256', 'typ' => 'JWT']);
    $base64Header = base64UrlEncode($header);
    $base64Payload = base64UrlEncode(json_encode($payload));
    $signature = hash_hmac('sha256', "$base64Header.$base64Payload", $secret, true);
    $base64Signature = base64UrlEncode($signature);
    return "$base64Header.$base64Payload.$base64Signature";
}

// گرفتن ورودی JSON
$data = json_decode(file_get_contents('php://input'), true);

if (empty($data['username']) || empty($data['password'])) {
    http_response_code(400);
    echo json_encode(['error' => 'نام کاربری و رمز عبور الزامی است.']);
    exit;
}

try {
    // جستجوی کاربر در دیتابیس
    $stmt = $conn->prepare("SELECT * FROM users WHERE username = :username LIMIT 1");
    $stmt->execute([':username' => $data['username']]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$user) {
        http_response_code(404);
        echo json_encode([
            'status' => 'error',
            'message' => 'کاربری با این نام یافت نشد.'
        ], JSON_UNESCAPED_UNICODE);
        exit;
    }

    if (!password_verify($data['password'], $user['password'])) {
        http_response_code(401);
        echo json_encode([
            'status' => 'error',
            'message' => 'رمز عبور اشتباه است.'
        ], JSON_UNESCAPED_UNICODE);
        exit;
    }

    // ایجاد Payload برای JWT
    $payload = [
        'iss' => JWT_ISSUER,
        'sub' => $user['id'],
        'username' => $user['username'],
        'iat' => time(),
        'exp' => time() + JWT_EXPIRE
    ];

    // ساخت توکن
    $token = createJWT($payload, JWT_SECRET);

    http_response_code(200);
    echo json_encode([
        'message' => 'ورود موفقیت‌آمیز بود.',
        'token' => $token,
        'success' => true,
        'user' => [
            'id' => $user['id'],
            'username' => $user['username'],
            'email' => $user['email'],
            'name' => $user['name'] ?? null,
            'role' => $user['role'] ?? null
        ]
    ]);

} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => 'خطای سرور: ' . $e->getMessage()]);
}